// Background service worker

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'fetchSource') {
        fetchUrlSource(request.url, sendResponse);
        return true; // Indicates we wish to send a response asynchronously
    }
    if (request.action === 'apiVerify') {
        postData(request.url, request.data, sendResponse);
        return true;
    }
    if (request.action === 'verificationFinished') {
        chrome.action.setBadgeText({ text: '', tabId: sender.tab.id });
        return false;
    }
});

chrome.action.onClicked.addListener((tab) => {
    // Show loading state on icon
    chrome.action.setBadgeText({ text: '...', tabId: tab.id });
    chrome.action.setBadgeBackgroundColor({ color: '#f39c12', tabId: tab.id }); // Orange/Yellow for processing

    // Trigger verification in content script
    chrome.tabs.sendMessage(tab.id, { action: 'startVerification' });
});

async function fetchUrlSource(url, sendResponse) {
    try {
        const response = await fetch(url, {
            method: 'GET',
            cache: 'reload', // Force a fresh fetch, ignoring local cache
            credentials: 'include', // Send cookies to maintain session and language settings
            // Headers are handled by the browser to match current environment
        });

        if (!response.ok) {
            sendResponse({ success: false, error: `HTTP error ${response.status}: ${response.statusText}` });
            return;
        }

        const text = await response.text();
        sendResponse({ success: true, data: text });

    } catch (error) {
        sendResponse({ success: false, error: error.toString() });
    }
}

async function postData(url, data, sendResponse) {
    try {
        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(data)
        });

        if (!response.ok) {
            sendResponse({ success: false, error: `HTTP error ${response.status}: ${response.statusText}` });
            return;
        }

        const json = await response.json();

        // Version check
        const manifest = chrome.runtime.getManifest();
        const currentVersion = manifest.version;
        const requiredVersion = json.extension_version;

        if (requiredVersion && isVersionOlder(currentVersion, requiredVersion)) {
            json.update_required = true;
            json.required_version = requiredVersion;
        }

        sendResponse({ success: true, data: json });

    } catch (error) {
        sendResponse({ success: false, error: error.toString() });
    }
}

function isVersionOlder(current, required) {
    const c = current.split('.').map(Number);
    const r = required.split('.').map(Number);
    for (let i = 0; i < Math.max(c.length, r.length); i++) {
        const cv = c[i] || 0;
        const rv = r[i] || 0;
        if (cv < rv) return true;
        if (cv > rv) return false;
    }
    return false;
}
