// Content script to run on any page
(function () {
    console.log("[SignedText] content.js loaded on:", window.location.href);
    const hostname = window.location.hostname;
    const isSignedText = hostname.endsWith('signedtext.com');
    const isLocalhost = hostname === 'localhost' || hostname === '127.0.0.1';

    // Domain for API calls - defaults to signedtext.com
    let apiDomain = 'https://signedtext.com';

    // Check storage for API setting
    chrome.storage.sync.get({ enableLocalhost: false, useLocalhostApi: false, preferredLanguage: 'en' }, (items) => {
        console.log("[SignedText] Storage settings loaded:", items);
        if (items.useLocalhostApi) {
            apiDomain = 'http://localhost';
        }

        const helperLang = getUrlLanguage();
        const verificationLang = items.preferredLanguage;

        // Run existing logic strictly on signedtext/localhost
        if (isSignedText || isLocalhost) {
            if (isLocalhost) {
                if (items.enableLocalhost) {
                    document.body.classList.add('signedtext-extension-active');
                    initExtension(helperLang);
                }
            } else {
                // signedtext.com - always run
                document.body.classList.add('signedtext-extension-active');
                initExtension(helperLang);
            }
        }

        // Message listener for manual trigger
        chrome.runtime.onMessage.addListener((message) => {
            if (message.action === 'startVerification') {
                scanForSignedBlocks(verificationLang);
            }
        });
    });

    // SCAN LOGIC
    function scanForSignedBlocks(lang) {
        try {
            console.log("[SignedText] Scanning for signed blocks...");

            // Domain for signature markers is always signedtext.com (as per user request)
            // regardless of where we send the verification request.
            let domain = 'signedtext.com';

            // Escape for regex
            const escapeRegExp = (string) => string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const domainPattern = escapeRegExp(domain);

            // Regex structure from SourceAbstract.php:
            // ~(https?:\/\/)?(www\.)?DOMAIN(\/[a-zA-Z]{2})?\/code\/([a-zA-Z]{10})(\.v[0-9]+)?(\.[0-9]+)?(#start|#end)?~iu
            // JS RegExp:
            // We use global flag 'g' to match all occurrences in a text node
            const regex = new RegExp(`(https?:\\/\\/)?(www\\.)?${domainPattern}(\\/[a-zA-Z]{2})?\\/code\\/([a-zA-Z]{10})(?:\\.v([0-9]+))?(?:\\.([0-9]+))?(#start|#end)?`, 'gi');

            const starts = {}; // Map CODE.vX.Y -> Node
            const ends = [];   // List of { key, node }

            // 1. Scan ANCHORS specifically
            const anchors = document.querySelectorAll('a');
            anchors.forEach(el => {
                let text = el.href;
                processText(text, el, true);
            });

            // 2. Scan ALL TEXT NODES using TreeWalker
            const walker = document.createTreeWalker(
                document.body,
                NodeFilter.SHOW_TEXT,
                {
                    acceptNode: function (node) {
                        if (node.textContent.includes('code/')) {
                            return NodeFilter.FILTER_ACCEPT;
                        }
                        return NodeFilter.FILTER_SKIP;
                    }
                }
            );

            let currentNode;
            while (currentNode = walker.nextNode()) {
                processText(currentNode.textContent, currentNode, false);
            }

            function processText(text, node, isElement) {
                if (!text) return;

                let match;
                regex.lastIndex = 0;

                while ((match = regex.exec(text)) !== null) {
                    const code = match[4].toUpperCase();
                    const version = match[5] || null;
                    const num = match[6] || "";
                    const type = (match[7] || '').toLowerCase(); // #start or #end

                    const key = `${code}.v${version || '?'}.${num || '?'}`;

                    // Calculate offsets for TextNodes
                    // For #start, content begins AFTER the match
                    const startOffset = match.index + match[0].length;
                    // For #end, content ends BEFORE the match
                    const endOffset = match.index;

                    if (type === '#start') {
                        if (!starts[key]) {
                            starts[key] = {
                                node: node,
                                code, version, num,
                                isElement: isElement,
                                offset: startOffset
                            };
                        }
                    } else if (type === '#end') {
                        ends.push({
                            key,
                            code, version, num,
                            node: node,
                            isElement: isElement,
                            offset: endOffset
                        });
                    }
                }
            }

            const blocksToVerify = [];

            // Group markers by key
            const pendingBlocks = {};

            function getOrCreateBlock(key, code, version, num) {
                if (!pendingBlocks[key]) {
                    pendingBlocks[key] = {
                        code, version, num,
                        start: null,
                        end: null
                    };
                }
                return pendingBlocks[key];
            }

            // Process all starts
            for (let key in starts) {
                const s = starts[key];
                const block = getOrCreateBlock(key, s.code, s.version, s.num);
                block.start = s;
            }

            // Process all ends
            ends.forEach(e => {
                const block = getOrCreateBlock(e.key, e.code, e.version, e.num);
                block.end = e;
            });

            // Form ranges and prepare for verification
            for (let key in pendingBlocks) {
                const b = pendingBlocks[key];
                if (b.start && b.end) {
                    try {
                        const range = document.createRange();

                        // Set Start
                        if (b.start.isElement) {
                            range.setStartAfter(b.start.node);
                        } else {
                            range.setStart(b.start.node, b.start.offset);
                        }

                        // Set End
                        if (b.end.isElement) {
                            range.setEndBefore(b.end.node);
                        } else {
                            range.setEnd(b.end.node, b.end.offset);
                        }

                        const fragment = range.cloneContents();
                        const div = document.createElement('div');
                        div.appendChild(fragment);

                        blocksToVerify.push({
                            code: b.code,
                            version: b.version,
                            num: b.num,
                            content: div.innerHTML,
                            content_text: div.innerText,
                            element: b.end.node // Use end node for positioning UI
                        });
                    } catch (e) {
                        console.error("Error extracting block content for " + key, e);
                    }
                }
            }

            if (blocksToVerify.length > 0) {
                verifyBlocks(blocksToVerify, lang);
            }
        } catch (err) {
            console.error("[SignedText] Critical error during scan:", err);
        }
    }

    function verifyBlocks(blocks, lang) {
        // Fetch raw source first to ensure faithful HTML (as per user request)
        chrome.runtime.sendMessage({
            action: 'fetchSource',
            url: window.location.href
        }, (sourceResponse) => {
            const faithfulContent = (sourceResponse && sourceResponse.success) ? sourceResponse.data : document.documentElement.outerHTML;

            const payload = {
                html: faithfulContent,
                copy: document.body.innerText,
                url: window.location.href,
                lang: lang,
                blocks: blocks.map(b => ({
                    code: b.code,
                    version: b.version,
                    num: b.num
                }))
            };

            chrome.runtime.sendMessage({
                action: 'apiVerify',
                url: `${apiDomain}/api/verify`,
                data: payload
            }, (response) => {
                if (response && response.success && response.data && response.data.results) {
                    displayResults(blocks, response.data.results, lang);

                    if (response.data.update_required) {
                        showUpdateNotification(response.data.required_version, lang);
                    }
                } else {
                    console.error("Verification failed", response);
                }
                // Notify background script that verification is finished (to clear badge)
                chrome.runtime.sendMessage({ action: 'verificationFinished' });
            });
        });
    }

    function displayResults(blocks, results, lang) {
        // Group results by code for flexible matching
        const apiByCode = {};
        results.forEach(res => {
            if (!apiByCode[res.code]) apiByCode[res.code] = [];
            apiByCode[res.code].push(res);
        });

        blocks.forEach(block => {
            const codeResults = apiByCode[block.code] || [];
            let groupResults = [];

            if (codeResults.length > 0) {
                // 1. Try exact match (code + num)
                groupResults = codeResults.filter(res => String(res.num || "") === String(block.num || ""));

                // 2. Fallback: If no exact match, and both are "empty-ish" (e.g. "" vs "0" or "1"), and it's unambiguous
                if (groupResults.length === 0) {
                    const isSimplified = (n) => !n || n === "0" || n === "1";
                    if (isSimplified(block.num)) {
                        // Find all potential "primary" results for this code
                        const primaryResults = codeResults.filter(res => isSimplified(res.num));
                        if (primaryResults.length > 0) {
                            groupResults = primaryResults;
                        }
                    }
                }

                // 3. Last fallback: if we only have one result for this code, use it
                if (groupResults.length === 0 && codeResults.length === 1) {
                    groupResults = codeResults;
                }
            }

            showResultUI(block, groupResults, lang);
        });
    }

    function escapeHtml(text) {
        if (!text) return text;
        return String(text)
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    function showResultUI(block, groupResults, lang) {
        const container = document.createElement('div');
        container.classList.add('signedtext-result-container');
        container.style.display = 'inline-block';
        container.style.marginLeft = '10px';
        container.style.padding = '5px 10px';
        container.style.borderRadius = '5px';
        container.style.fontSize = '12px';
        container.style.fontFamily = 'sans-serif';
        container.style.zIndex = '9999';
        container.style.position = 'relative';
        container.style.boxShadow = '0 1px 3px rgba(0,0,0,0.2)';

        // Logic for OK status (OR logic as requested)
        // res.status is exactly what VerifyLogicTrait returns (true/false)
        // We only use non-extraneous blocks for the badge status
        const isValid = groupResults.length > 0 && groupResults.some(res => res.status === true && !res.extraneous);
        const hasOnlineOk = groupResults.length > 0 && groupResults.some(res => res.status === true && res.sig_type === 'PAGE' && !res.extraneous);
        const hasCopyOk = groupResults.length > 0 && groupResults.some(res => res.status === true && !res.extraneous);

        if (isValid) {
            container.style.backgroundColor = '#dff0d8';
            container.style.border = '1px solid #d6e9c6';
            container.style.color = '#3c763d';

            let label = 'Verified';
            if (hasOnlineOk) {
                label = 'Verified online';
            } else if (hasCopyOk) {
                label = 'Verified copy';
            }

            container.innerHTML = `<strong>&#10004; ${_t(lang, label)}</strong>`;
        } else {
            container.style.backgroundColor = '#f2dede';
            container.style.border = '1px solid #ebccd1';
            container.style.color = '#a94442';
            container.innerHTML = `<strong>&#10008; ${_t(lang, 'Verification Failed')}</strong>`;
        }

        const details = document.createElement('div');
        details.style.display = 'none';
        details.style.position = 'absolute';
        details.style.backgroundColor = 'white';
        details.style.border = '1px solid #ccc';
        details.style.padding = '10px';
        details.style.borderRadius = '3px';
        details.style.marginTop = '5px';
        details.style.left = '0';
        details.style.top = '100%';
        details.style.width = '300px';
        details.style.color = '#333';
        details.style.textAlign = 'left';

        let detailsHtml = '';
        if (groupResults.length === 0) {
            detailsHtml = `<div style="font-weight:bold; margin-bottom: 5px;">Block: ${escapeHtml(block.code)}</div>`;
            detailsHtml += `<div style="color:red; font-style: italic;">${_t(lang, 'No signature data found for this block.')}</div>`;
        } else {
            groupResults.forEach((res, idx) => {
                if (idx > 0) detailsHtml += '<hr style="margin: 10px 0; border: 0; border-top: 1px solid #eee;">';

                detailsHtml += `<div style="font-weight:bold; margin-bottom: 5px;">Block: ${escapeHtml(res.block_name || block.code)} (${res.type})</div>`;
                if (res.extraneous) {
                    detailsHtml += `<div style="color: #888; font-style: italic; font-size: 11px; margin-bottom: 5px;">${_t(lang, 'Extraneous for signature')}: ${escapeHtml(res.sig_code)}</div>`;
                }
                if (res.signer_name) {
                    detailsHtml += `<div>Signer: ${escapeHtml(res.signer_name)}</div>`;
                }
                if (res.msg) {
                    detailsHtml += `<div style="font-style: italic; margin-top: 5px; margin-bottom: 5px; color: #555;">${escapeHtml(res.msg)}</div>`;
                }

                if (res.PAGE_STATUS === true) {
                    detailsHtml += `<div style="color:green">&#10004; ${escapeHtml(res.PAGE || _t(lang, 'Identical found in online source'))}</div>`;
                } else if (res.PAGE_STATUS === false) {
                    detailsHtml += `<div style="color:red">&#10008; ${escapeHtml(res.PAGE || _t(lang, 'Mismatch in online source'))}</div>`;
                }

                if (res.PASTEBIN_STATUS === true) {
                    detailsHtml += `<div style="color:green">&#10004; ${escapeHtml(res.PASTEBIN || _t(lang, 'Identical found in copy&paste source'))}</div>`;
                } else if (res.PASTEBIN_STATUS === false) {
                    detailsHtml += `<div style="color:red">&#10008; ${escapeHtml(res.PASTEBIN || _t(lang, 'Mismatch in copy&paste source'))}</div>`;
                }

                if (res.reasons && res.reasons.length > 0) {
                    const ulStyle = 'margin: 5px 0 5px 20px !important; padding: 0 !important; list-style: disc outside !important; text-align: left !important; background: none !important; display: block !important; border: none !important;';
                    const liStyle = 'margin: 2px 0 !important; padding: 0 !important; text-align: left !important; color: #333 !important; font-size: 12px !important; line-height: 1.4 !important; display: list-item !important; border: none !important; background: none !important;';
                    detailsHtml += `<div>Reasons:<ul style="${ulStyle}">${res.reasons.map(r => `<li style="${liStyle}">${escapeHtml(r)}</li>`).join('')}</ul></div>`;
                }
            });
        }

        details.innerHTML = detailsHtml;
        container.appendChild(details);

        const BASE_Z_INDEX = 9999;
        const HOVER_Z_INDEX = 10000;

        container.onmouseenter = () => {
            container.style.zIndex = HOVER_Z_INDEX;
            details.style.display = 'block';
        };
        container.onmouseleave = () => {
            container.style.zIndex = BASE_Z_INDEX;
            details.style.display = 'none';
        };

        if (block.element.nextSibling) {
            block.element.parentNode.insertBefore(container, block.element.nextSibling);
        } else {
            block.element.parentNode.appendChild(container);
        }
    }

    function showUpdateNotification(requiredVersion, lang) {
        if (document.getElementById('signedtext-update-banner')) return;

        const banner = document.createElement('div');
        banner.id = 'signedtext-update-banner';
        banner.style.position = 'fixed';
        banner.style.top = '0';
        banner.style.left = '0';
        banner.style.width = '100%';
        banner.style.backgroundColor = '#fff3cd';
        banner.style.color = '#856404';
        banner.style.padding = '10px 20px';
        banner.style.textAlign = 'center';
        banner.style.zIndex = '100000';
        banner.style.borderBottom = '1px solid #ffeeba';
        banner.style.fontSize = '14px';
        banner.style.fontWeight = 'bold';
        banner.style.boxShadow = '0 2px 5px rgba(0,0,0,0.1)';

        const installLink = `https://signedtext.com/${lang}/info/extension-install/`;

        banner.innerHTML = `
            ${_t(lang, 'A new version of SignedText Helper (%s) is available.', requiredVersion)} 
            <a href="${installLink}" target="_blank" style="color: #856404; text-decoration: underline; margin-left:10px;">
                ${_t(lang, 'Please reinstall the extension here.')}
            </a>
            <button style="margin-left: 20px; cursor: pointer; border: none; background: none; font-size: 16px; color: #856404;" onclick="this.parentElement.remove()">&#10006;</button>
        `;

        document.body.prepend(banner);
    }


    // EXISTING LOGIC for SignedText.com pages
    const translations = {
        'cs': {
            'Load Source Code (Extension)': 'Načíst zdrojový kód (Rozšíření)',
            'Loading...': 'Načítám...',
            'Requesting %s ...': 'Provádím požadavek na %s ...',
            'Loaded (%s chars).': 'Načteno (%s znaků).',
            'Error: %s': 'Chyba: %s',
            'Unknown error': 'Neznámá chyba',
            'Verified': 'Ověřeno',
            'Verified online': 'Ověřeno online',
            'Verified copy': 'Ověřená kopie',
            'Extraneous for signature': 'Extraneous k podpisu',
            'Verification Failed': 'Ověření selhalo',
            'No signature data found for this block.': 'Pro tento blok nebyla nalezena žádná data o podpisu.',
            'Identical found in online source': 'Identický obsah v online zdroji',
            'Mismatch in online source': 'Neshoda v online zdroji',
            'Identical found in copy&paste source': 'Identický obsah v copy&paste zdroji',
            'Mismatch in copy&paste source': 'Neshoda v copy&paste zdroji',
            'A new version of SignedText Helper (%s) is available.': 'Je k dispozici nová verze SignedText Helper (%s).',
            'Please reinstall the extension here.': 'Prosím, nainstalujte si novou verzi zde.'
        },
        'sk': {
            'Load Source Code (Extension)': 'Načítať zdrojový kód (Rozšírenie)',
            'Loading...': 'Načítavam...',
            'Requesting %s ...': 'Vykonávam požiadavku na %s ...',
            'Loaded (%s chars).': 'Načítané (%s znakov).',
            'Error: %s': 'Chyba: %s',
            'Unknown error': 'Neznáma chyba',
            'Verified': 'Overené',
            'Verified online': 'Overené online',
            'Verified copy': 'Overená kópia',
            'Extraneous for signature': 'Extraneous k podpisu',
            'Verification Failed': 'Overenie zlyhalo',
            'No signature data found for this block.': 'Pre tento blok neboli nájdené žiadne údaje o podpise.',
            'Identical found in online source': 'Identický obsah v online zdroji',
            'Mismatch in online source': 'Neshoda v online zdroji',
            'Identical found in copy&paste source': 'Identický obsah v copy&paste zdroji',
            'Mismatch in copy&paste source': 'Neshoda v copy&paste zdroji',
            'A new version of SignedText Helper (%s) is available.': 'Je k dispozícii nová verzia SignedText Helper (%s).',
            'Please reinstall the extension here.': 'Prosím, nainštalujte si novú verziu tu.'
        }
    };

    function getUrlLanguage() {
        const path = window.location.pathname;
        if (path.startsWith('/sk')) return 'sk';
        if (path.startsWith('/cs')) return 'cs';
        return 'en';
    }

    function _t(lang, str, ...args) {
        let translated = str;
        if (translations[lang] && translations[lang][str]) {
            translated = translations[lang][str];
        } else if (str === 'A new version of SignedText Helper (%s) is available.') {
            // Default English if not translated
            translated = 'A new version of SignedText Helper (%s) is available.';
        } else if (str === 'Please reinstall the extension here.') {
            translated = 'Please reinstall the extension here.';
        }

        let i = 0;
        return translated.replace(/%s/g, () => args[i++] || '');
    }

    function initExtension(lang) {
        const textareas = document.getElementsByTagName('textarea');
        if (textareas.length === 0) return;
        const textarea = textareas[0];
        let targetUrl = null;
        const links = document.querySelectorAll('a[target="_blank"]');
        for (const link of links) {
            const href = link.href;
            const text = link.textContent.trim();
            if (text.startsWith('http') && (href.startsWith(text) || text.startsWith(href))) {
                targetUrl = href;
                break;
            }
            const parent = link.parentElement;
            if (!parent) continue;
            const parentText = parent.textContent;
            if (parentText.includes('URL:')) {
                targetUrl = href;
                break;
            }
        }
        if (!targetUrl) return;
        const container = document.createElement('div');
        container.style.marginTop = '10px';
        container.style.marginBottom = '10px';
        const btn = document.createElement('button');
        btn.innerHTML = '&#8681; ' + _t(lang, 'Load Source Code (Extension)') + ' &#8681;';
        btn.className = 'btn btn-success btn-block';
        btn.style.fontWeight = 'bold';
        btn.title = _t(lang, 'Load Source Code (Extension)') + ': ' + targetUrl;
        const statusMsg = document.createElement('div');
        statusMsg.style.marginTop = '5px';
        statusMsg.style.fontStyle = 'italic';
        statusMsg.style.color = '#555';
        btn.onclick = (e) => {
            e.preventDefault();
            btn.disabled = true;
            btn.innerHTML = _t(lang, 'Loading...');
            statusMsg.textContent = _t(lang, 'Requesting %s ...', targetUrl);
            statusMsg.style.color = '#555';
            chrome.runtime.sendMessage({
                action: 'fetchSource',
                url: targetUrl
            }, (response) => {
                btn.disabled = false;
                btn.innerHTML = '&#8681; ' + _t(lang, 'Load Source Code (Extension)') + ' &#8681;';
                if (response && response.success) {
                    textarea.value = response.data;
                    textarea.dispatchEvent(new Event('change', { bubbles: true }));
                    textarea.dispatchEvent(new Event('input', { bubbles: true }));
                    statusMsg.textContent = _t(lang, 'Loaded (%s chars).', response.data.length);
                    statusMsg.style.color = 'green';
                    textarea.style.backgroundColor = '#e8f5e9';
                    setTimeout(() => textarea.style.backgroundColor = '', 1000);
                } else {
                    statusMsg.textContent = _t(lang, 'Error: %s', (response ? response.error : _t(lang, 'Unknown error')));
                    statusMsg.style.color = 'red';
                }
            });
        };
        container.appendChild(btn);
        container.appendChild(statusMsg);
        textarea.parentNode.insertBefore(container, textarea);
    }
})();
